/*****************************************************************************
* FILE:         spi_tclMySPINSndCtrlMonitor.cpp
* PROJECT:      G3G project
* SW-COMPONENT: Smart Phone Integration
*----------------------------------------------------------------------------
*
* DESCRIPTION:  DBus proxy for audio manager based on g3g_dbusobjectproxy
*
*----------------------------------------------------------------------------
* COPYRIGHT:      &copy; BSOT
*
* HISTORY:
* Date       |  Author                          | Modifications
* 28.04.2015 |  L.Weise (BSOT/ENG - wel2hi)     | Added udev monitor to detect
                                                  sample rate changes
* 21.05.2015 |  L.Weise (BSOT/ENG - wel2hi)     | bugfix: end monitor thread in
*                                                 case audio streaming is switched
*                                                 off.
*
******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "spi_tclMySPINSndCtrlMonitorIF.h"
#include "spi_tclMySPINSndCtrlMonitor.h"
#include "mspin_logging.h"


//#define OSAL_S_IMPORT_INTERFACE_GENERIC
//#include "osal_if.h"

#include <unistd.h>    // usleep()
#include <pthread.h>   // pthread_create

using namespace std;

bool spi_tclMySPINSndCtrlMonitor::bInitCaptureEnabled = false;
bool spi_tclMySPINSndCtrlMonitor::bInitDone = false;

/*******************************************************************************
*
* FUNCTION: spi_tclMySPINSndCtrlMonitor::
*             spi_tclMySPINSndCtrlMonitor()
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class
*
* PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
*
* RETURNVALUE: None.
*
*******************************************************************************/
/***************************************************************************
** FUNCTION:  spi_tclMySPINSndCtrlMonitor::spi_tclMySPINSndCtrlMonitor()
***************************************************************************/
/*!
* \fn      spi_tclMySPINSndCtrlMonitor( )
* \brief   Constructor.
*
*
**************************************************************************/
spi_tclMySPINSndCtrlMonitor::spi_tclMySPINSndCtrlMonitor( )
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );


   _u32SampleRate       = 44100;   // sample rate fix for AOAP and BT audio; changes for iAP
   _bEndPointEnabled    = false;   // 0/1 in case endpoint disabled/enabled.
   _ptrMonitorCallback  = NULL;
   _hMonitorThreadId    = 0;
   _bTerminateMonitorThread = false;
   _bMonitorThreadCreated = false;

}

/***************************************************************************
** FUNCTION:  spi_tclMySPINSndCtrlMonitor::~spi_tclMySPINSndCtrlMonitor
***************************************************************************/
/*!
* \fn      spi_tclMySPINSndCtrlMonitor( )
* \brief   Destructor.
*
**************************************************************************/
spi_tclMySPINSndCtrlMonitor::~spi_tclMySPINSndCtrlMonitor(void)
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );
   _ptrMonitorCallback  = NULL;
}

/***************************************************************************
** FUNCTION:  void spi_tclMySPINSndCtrlMonitor::vInitMonitor()
***************************************************************************/
/*!
* \fn      vInitMonitor( )
* \brief   Triggers monitor thread creation and sets callback for sample
*          rate changes. The monitor thread listens for messages in case
*          the sample rate changes.
*
**************************************************************************/
void spi_tclMySPINSndCtrlMonitor::vInitMonitor(spi_tclMySPINSndCtrlMonitorIF* ptrMonitorCallback)
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

   vSetMonitorCallback(ptrMonitorCallback);

   // only create one thread (in case of TTFis command)
   if ( 0 == _hMonitorThreadId )
   {
       bInitCaptureEnabled = false;
       bInitDone = false;

       vMonitorThreadSetup();
   }
   else
   {
      mspin_log_printLn(eMspinVerbosityInfo, "%s() Thread already started _hMonitorThreadId: 0x%X.",__PRETTY_FUNCTION__, _hMonitorThreadId );
   }
}

/***************************************************************************
** FUNCTION:  void spi_tclMySPINSndCtrlMonitor::vStopMonitor()
***************************************************************************/
/*!
* \fn      vStopMonitor( )
* \brief   Turns off the monitor loop and exits thread.
*
**************************************************************************/
void spi_tclMySPINSndCtrlMonitor::vStopMonitor()
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );
   _bTerminateMonitorThread = true;

   if(_bMonitorThreadCreated)
   {
       pthread_join(_hMonitorThreadId, NULL);
       _bMonitorThreadCreated = false;
       mspin_log_printLn(eMspinVerbosityInfo, "%s() _hMonitorThread is joined successfully.",__PRETTY_FUNCTION__ );
   }
}


/***************************************************************************
** FUNCTION:  bool spi_tclMySPINSndCtrlMonitor::bGetCaptureEnabled()
***************************************************************************/
/*!
* \fn      bGetCaptureEnabled( )
* \brief   Check whether capture device was enabled after monitor init
*
**************************************************************************/
bool spi_tclMySPINSndCtrlMonitor::bGetInitCaptureEnabled()
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    while (!bInitDone){
        usleep(5000);            // wait for 5 ms
    };

    mspin_log_printLn(eMspinVerbosityInfo, "%s() EXIT",__PRETTY_FUNCTION__ );
    return bInitCaptureEnabled;
}


/***************************************************************************
** FUNCTION:  void spi_tclMySPINSndCtrlMonitor::vSetMonitorCallback()
***************************************************************************/
/*!
* \fn      vSetMonitorCallback( )
* \brief   Called to set callback function for sample rate change.
*
* \param   [IN] ptrMonitorCallback: pointer to callback function
*
**************************************************************************/
void spi_tclMySPINSndCtrlMonitor::vSetMonitorCallback(spi_tclMySPINSndCtrlMonitorIF* ptrMonitorCallback)
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

   _ptrMonitorCallback = ptrMonitorCallback;
}

/***************************************************************************
** FUNCTION:  void spi_tclMySPINSndCtrlMonitor::vFireMonitorCallback()
***************************************************************************/
/*!
* \fn      vFireMonitorCallback( )
* \brief   Called by monitor thread in case of sample rate change. Calls
*          callback function from AudioManager and passes sample rate.
*
* \param   [IN] u32SampleRate: sample rate of currently played title (iAP2 only)
* \param   [IN] bEndPointEnabled: true sample rate is active
*
**************************************************************************/
void spi_tclMySPINSndCtrlMonitor::vFireMonitorCallback(U32 u32SampleRate, bool bEndPointEnabled)
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

   if (_ptrMonitorCallback != NULL)
   {
      _ptrMonitorCallback->vSampleRateChanged_CB(u32SampleRate,bEndPointEnabled);
   }
   else
   {
      mspin_log_printLn(eMspinVerbosityError, "%s() - FAILURE: _ptrMonitorCallback = NULL.",__PRETTY_FUNCTION__ );
   }

}

/***************************************************************************
** FUNCTION:  void spi_tclMySPINSndCtrlMonitor::_vMonitorThreadSetup()
***************************************************************************/
/*!
* \fn      _vMonitorThreadSetup( )
* \brief   Creates monitor thread. The monitor thread listens for
*          messages in case the sample rate changes.
*
**************************************************************************/
void spi_tclMySPINSndCtrlMonitor::vMonitorThreadSetup()
{
   mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

   // now start the monitor thread
   //wel2hi todo: check for u32Priority,s32StackSize,...

   S32 err = -1;

   _bTerminateMonitorThread = false;
   err = pthread_create(&_hMonitorThreadId, NULL, (void *(*)(void *))&(spi_tclMySPINSndCtrlMonitor::vMonitorThreadFunction), (void *)this);
   if (0 == err)
   {
      _bMonitorThreadCreated = true;
      mspin_log_printLn(eMspinVerbosityInfo, "%s() - SUCCESS: Created thread %s.",__PRETTY_FUNCTION__, "SPI_SCT" );

      //Set thread name
      err = pthread_setname_np(_hMonitorThreadId, (const char*)"SPI_SCT");
      if (0 == err)
      {
         mspin_log_printLn(eMspinVerbosityInfo, "%s() - SUCCESS: Set thread name thread %s.",__PRETTY_FUNCTION__, "SPI_SCT" );
      }
      else
      {
         mspin_log_printLn(eMspinVerbosityError, "%s() - ERROR: Could not set thread name!",__PRETTY_FUNCTION__);
      }
   }
   else
   {
       _bMonitorThreadCreated = false;
       mspin_log_printLn(eMspinVerbosityFatal, "%s() - ERROR: Could not create monitor thread!",__PRETTY_FUNCTION__);
   }

}



S32 spi_tclMySPINSndCtrlMonitor::s32ElemVal(struct uac2_ctrl *ctrl, U32 u32Index)
{
    return ctrl->elems_val[u32Index];
}

S32 spi_tclMySPINSndCtrlMonitor::s32ElemUpdate(struct uac2_ctrl *ctrl, U32 u32Index)
{
    S32 err;
    snd_ctl_elem_value_t **elem = &ctrl->elems[u32Index];
    if ((err = snd_ctl_elem_read(ctrl->ctl, *elem)) < 0) {
        mspin_log_printLn(eMspinVerbosityFatal, "%s() ERROR: Unable to read contents: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
    }
    ctrl->elems_val[u32Index] = snd_ctl_elem_value_get_integer(*elem, 0);

    mspin_log_printLn(eMspinVerbosityInfo, "%s() updated u32Index %d to val %ld",__PRETTY_FUNCTION__, u32Index,ctrl->elems_val[u32Index] );
    return err;
}

S32 spi_tclMySPINSndCtrlMonitor::s32ElemInit(struct uac2_ctrl *ctrl, U32 u32Index, const char * name)
{
    S32 err;

    snd_ctl_elem_value_t **elem = &ctrl->elems[u32Index];
    if ((err = snd_ctl_elem_value_malloc(elem)) < 0) {
        mspin_log_printLn(eMspinVerbosityFatal, "%s() ERROR: Unable to malloc: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
    }
    snd_ctl_elem_value_set_interface(*elem, SND_CTL_ELEM_IFACE_CARD);
    snd_ctl_elem_value_set_name(*elem, name);
    return s32ElemUpdate(ctrl, u32Index);
}

void spi_tclMySPINSndCtrlMonitor::vElemFree(struct uac2_ctrl *ctrl, U32 u32Index )
{
    snd_ctl_elem_value_t *elem = ctrl->elems[u32Index];

    if( NULL != elem )
    {
        snd_ctl_elem_value_free( elem );
    }
}

S32 spi_tclMySPINSndCtrlMonitor::s32SndCtrlInit(struct uac2_ctrl *ctrl)
{
    S32 err;

    ctrl->name = "hw:UAC2Gadget";
    err = snd_ctl_open(&ctrl->ctl, ctrl->name, SND_CTL_READONLY/*|SND_CTL_NONBLOCK*/);
    if(err)
    {
        ctrl->ctl = NULL;
        return err;
    }
    err = snd_ctl_subscribe_events(ctrl->ctl, 1);
    if(err)
    {
        snd_ctl_close(ctrl->ctl);
        ctrl->ctl = NULL;
        return err;
    }
    s32ElemInit(ctrl, CTRL_UAC2_ELEM_P_RATE, SND_CTL_NAME_PLAYBACK"sample rate");
    s32ElemInit(ctrl, CTRL_UAC2_ELEM_C_RATE, SND_CTL_NAME_CAPTURE"sample rate");
    s32ElemInit(ctrl, CTRL_UAC2_ELEM_P_ENABLED, SND_CTL_NAME_PLAYBACK"enabled");
    s32ElemInit(ctrl, CTRL_UAC2_ELEM_C_ENABLED, SND_CTL_NAME_CAPTURE"enabled");
    return err;
}



S32 spi_tclMySPINSndCtrlMonitor::s32SndCtrlDeInit(struct uac2_ctrl *ctrl)
{
    S32 err = 0;

    if( NULL == ctrl )
    {
        err = -1;
    }
    else
    {
        vElemFree(ctrl, CTRL_UAC2_ELEM_P_RATE );
        vElemFree(ctrl, CTRL_UAC2_ELEM_C_RATE );
        vElemFree(ctrl, CTRL_UAC2_ELEM_P_ENABLED );
        vElemFree(ctrl, CTRL_UAC2_ELEM_C_ENABLED );

        if( NULL != ctrl->ctl )
        {
            //if ((err = snd_ctl_subscribe_events(ctrl->ctl, 0)) <0) {
            //mspin_log_printLn(eMspinDemoVerbosityFatal, "%s() ERROR: Unable to unsubscribe: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
            //}

            snd_ctl_close(ctrl->ctl);
            ctrl->ctl = NULL;
        }
    }

    return err;
}



/***************************************************************************
** FUNCTION:  void spi_tclMySPINSndCtrlMonitor::vMonitorThreadFunction()
***************************************************************************/
/*!
* \fn      vMonitorThreadFunction( )
* \brief   Entry function of monitor thread. The monitor thread listens for
*          messages in case the sample rate changes.
*
* \param   [IN] pvArg: this pointer of class
*
**************************************************************************/
void spi_tclMySPINSndCtrlMonitor::vMonitorThreadFunction( void* pvArg )
{
    spi_tclMySPINSndCtrlMonitor*  poMyself;

    U32 u32CaptureSampleRate = 0;
    U32 u32PlaybackSampleRate = 0;
    U32 u32CaptureEnabled = 0;
    U32 u32PlaybackEnabled = 0;
    bool bCaptureEnabled = false;

    bool bReadFromSndCtl = false;

    S32 s32RetVal = 0;
    S32 s32SndCtlRetVal = 0;


    struct uac2_ctrl sCtrl;
    snd_ctl_event_type_t enSndCtlEventType;

    snd_ctl_event_t *posSndCtlEvent = NULL;

    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    memset( &sCtrl , 0 , sizeof( struct uac2_ctrl ) );

    poMyself = (spi_tclMySPINSndCtrlMonitor *) pvArg;

    if( poMyself == NULL )
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: poMyself is NULL!",__PRETTY_FUNCTION__ );
        s32RetVal = -1;
    }

    if( 0 == s32RetVal )
    {
        s32SndCtlRetVal = snd_ctl_event_malloc(&posSndCtlEvent);

        if( 0 != s32SndCtlRetVal )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: snd_ctl_event_malloc() returned error: %i",__PRETTY_FUNCTION__, s32SndCtlRetVal );
            posSndCtlEvent = NULL;
            s32RetVal = -2;
        }
        else if( NULL == posSndCtlEvent )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: posSndCtlEvent is NULL!",__PRETTY_FUNCTION__ );
            s32RetVal = -2;
        }
    }

    if( 0 == s32RetVal )
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s() Call s32SndCtrlInit()",__PRETTY_FUNCTION__ );

        s32SndCtlRetVal = s32SndCtrlInit(&sCtrl);
        if( 0 != s32SndCtlRetVal )
        {
            mspin_log_printLn(eMspinVerbosityFatal, "%s() ERROR: s32SndCtrlInit() failed",__PRETTY_FUNCTION__ );
            s32RetVal = -2;
        }else{

            u32CaptureEnabled = s32ElemVal(&sCtrl, CTRL_UAC2_ELEM_C_ENABLED);
            bInitCaptureEnabled = ( 0 != u32CaptureEnabled);
            if (bInitCaptureEnabled) {
                mspin_log_printLn(eMspinVerbosityInfo, "%s() Capture enabled (%i) after s32SndCtrlInit",__PRETTY_FUNCTION__, bInitCaptureEnabled );
            }else{
                mspin_log_printLn(eMspinVerbosityInfo, "%s() Capture disabled (%i) after s32SndCtrlInit",__PRETTY_FUNCTION__, bInitCaptureEnabled );
            }
        }
    }

    bInitDone = true;

    while( ( 0 == s32RetVal ) && (poMyself) &&
           ( false == poMyself->_bTerminateMonitorThread ) )
    {

        bReadFromSndCtl = false;

        s32SndCtlRetVal = snd_ctl_wait( sCtrl.ctl,
                500 /* timeout in ms */ );

        if( 0 > s32SndCtlRetVal )
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s() snd_ctl_wait() returned error: %d",__PRETTY_FUNCTION__, s32SndCtlRetVal );
            s32RetVal = 3;
        }
        else if( 0 == s32SndCtlRetVal )
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s() snd_ctl_wait() returned timeout...",__PRETTY_FUNCTION__ );
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s() snd_ctl_wait() returned: %d",__PRETTY_FUNCTION__, s32SndCtlRetVal );

            s32SndCtlRetVal = snd_ctl_read(sCtrl.ctl, posSndCtlEvent);

            if( 0 > s32SndCtlRetVal )
            {
                mspin_log_printLn(eMspinVerbosityInfo, "%s()  snd_ctl_read() returned error: %d",__PRETTY_FUNCTION__, s32SndCtlRetVal );
                s32RetVal = -4;
            }
            else if( 0 == s32SndCtlRetVal )
            {
                mspin_log_printLn(eMspinVerbosityInfo, "%s() snd_ctl_read() returned no events. Ignore",__PRETTY_FUNCTION__ );
            }
            else
            {
                enSndCtlEventType = snd_ctl_event_get_type(posSndCtlEvent);

                if (enSndCtlEventType != SND_CTL_EVENT_ELEM)
                {
                    mspin_log_printLn(eMspinVerbosityInfo, "%s() Ignore event of unhandled type %d",__PRETTY_FUNCTION__, enSndCtlEventType );
                }
                else
                {
                    bReadFromSndCtl = true;
                }
            }
        } /* else if( 0 == s32SndCtlRetVal ) */

        if( ( 0 == s32RetVal ) && ( true == bReadFromSndCtl ) )
        {
            s32ElemUpdate(&sCtrl, CTRL_UAC2_ELEM_C_RATE);
            u32CaptureSampleRate = s32ElemVal(&sCtrl, CTRL_UAC2_ELEM_C_RATE);

            s32ElemUpdate(&sCtrl, CTRL_UAC2_ELEM_P_RATE);
            u32PlaybackSampleRate = s32ElemVal(&sCtrl, CTRL_UAC2_ELEM_P_RATE);

            s32ElemUpdate(&sCtrl, CTRL_UAC2_ELEM_C_ENABLED);
            u32CaptureEnabled = s32ElemVal(&sCtrl, CTRL_UAC2_ELEM_C_ENABLED);

            s32ElemUpdate(&sCtrl, CTRL_UAC2_ELEM_P_ENABLED);
            u32PlaybackEnabled = s32ElemVal(&sCtrl, CTRL_UAC2_ELEM_P_ENABLED);

            bCaptureEnabled = ( 0 != u32CaptureEnabled);

            mspin_log_printLn(eMspinVerbosityInfo, "%s() CTRL_UAC2_ELEM_C_ENABLED = %d, CTRL_UAC2_ELEM_C_RATE = %d",__PRETTY_FUNCTION__, u32CaptureEnabled, u32CaptureSampleRate );
            mspin_log_printLn(eMspinVerbosityInfo, "%s() CTRL_UAC2_ELEM_P_ENABLED = %d, CTRL_UAC2_ELEM_P_RATE = %d",__PRETTY_FUNCTION__, u32PlaybackEnabled, u32PlaybackSampleRate );

            poMyself->vFireMonitorCallback(u32CaptureSampleRate,bCaptureEnabled);

        } /* if( ( 0 == s32RetVal ) && ( true == bReadFromSndCtl ) ) */

    } /* while( ( 0 == s32RetVal ) && ( false == poMyself->_bTerminateMonitorThread ) ) */


    mspin_log_printLn(eMspinVerbosityInfo, "%s() Cleanup",__PRETTY_FUNCTION__ );
    s32SndCtrlDeInit(&sCtrl);

    if( NULL != posSndCtlEvent )
    {
        snd_ctl_event_free( posSndCtlEvent );
        posSndCtlEvent = NULL;
    }

    mspin_log_printLn(eMspinVerbosityInfo, "%s() EXIT",__PRETTY_FUNCTION__ );

    pthread_exit(NULL);

}
